#region

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows;
using System.Windows.Forms;
using Hearthstone_Deck_Tracker.Controls.Error;
using Hearthstone_Deck_Tracker.Stats;
using Hearthstone_Deck_Tracker.Utility;
using Hearthstone_Deck_Tracker.Utility.Extensions;
using Hearthstone_Deck_Tracker.Utility.Logging;
using Hearthstone_Deck_Tracker.Utility.RemoteData;
using Hearthstone_Deck_Tracker.Windows;

#endregion

namespace Hearthstone_Deck_Tracker.FlyoutControls.Options.Tracker
{
	/// <summary>
	/// Interaction logic for OtherTracker.xaml
	/// </summary>
	public partial class TrackerSettings
	{
		private bool _initialized;

		public TrackerSettings()
		{
			InitializeComponent();
#if(SQUIRREL)
			CheckboxConfigSaveAppData.Visibility = Visibility.Collapsed;
			CheckboxDataSaveAppData.Visibility = Visibility.Collapsed;
			SelectSaveDataPath.Visibility = Visibility.Collapsed;
#endif
		}

		public void Load()
		{
			CheckboxCloseTray.IsChecked = Config.Instance.CloseToTray;
			CheckboxMinimizeTray.IsChecked = Config.Instance.MinimizeToTray;
			CheckboxStartMinimized.IsChecked = Config.Instance.StartMinimized;
			CheckboxCheckForUpdates.IsChecked = Config.Instance.CheckForUpdates;
			CheckboxCheckForBetaUpdates.IsChecked = Config.Instance.CheckForBetaUpdates;
			CheckboxCloseWithHearthstone.IsChecked = Config.Instance.CloseWithHearthstone;
			CheckboxStartHearthstoneWithHDT.IsChecked = Config.Instance.StartHearthstoneWithHDT;
			CheckboxAdvancedWindowSearch.IsChecked = Config.Instance.UseAnyUnityWindow;
			CheckBoxShowSplashScreen.IsChecked = Config.Instance.ShowSplashScreen;
			CheckboxStartWithWindows.IsChecked = Config.Instance.StartWithWindows;
			CheckBoxAnalytics.IsChecked = Config.Instance.GoogleAnalytics;

			CheckboxAlternativeScreenCapture.IsChecked = Config.Instance.AlternativeScreenCapture;
#if(!SQUIRREL)
			CheckboxConfigSaveAppData.IsChecked = Config.Instance.SaveConfigInAppData;
			CheckboxDataSaveAppData.IsChecked = Config.Instance.SaveDataInAppData;
#endif

			CheckboxShowNewsBar.IsChecked = null;

			ConfigWrapper.IgnoreNewsIdChanged += () =>
			{
				CheckboxShowNewsBar.IsChecked = ConfigWrapper.IgnoreNewsId == -1;
			};
			Remote.Config.Loaded += data =>
			{
				CheckboxShowNewsBar.IsChecked = Config.Instance.IgnoreNewsId < data?.News?.Id;
			};

			_initialized = true;
		}

		private void SaveConfig(bool updateOverlay)
		{
			Config.Save();
			if(updateOverlay)
				Core.Overlay.Update(true);
		}

		private void CheckboxMinimizeTray_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.MinimizeToTray = true;
			SaveConfig(false);
		}

		private void CheckboxMinimizeTray_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.MinimizeToTray = false;
			SaveConfig(false);
		}

		private void CheckboxCloseTray_Checked(object sender, RoutedEventArgs e)
		{
			if (!_initialized)
				return;
			Config.Instance.CloseToTray = true;
			SaveConfig(false);
		}

		private void CheckboxCloseTray_Unchecked(object sender, RoutedEventArgs e)
		{
			if (!_initialized)
				return;
			Config.Instance.CloseToTray = false;
			SaveConfig(false);
		}

		private void CheckboxStartMinimized_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.StartMinimized = true;
			SaveConfig(false);
		}

		private void CheckboxStartMinimized_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.StartMinimized = false;
			SaveConfig(false);
		}

		private void CheckboxCheckForUpdates_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CheckForUpdates = true;
			SaveConfig(false);
		}

		private void CheckboxCheckForUpdates_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CheckForUpdates = false;
			SaveConfig(false);
		}

		private void CheckboxCloseWithHearthstone_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CloseWithHearthstone = true;
			Config.Save();
		}

		private void CheckboxCloseWithHearthstone_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CloseWithHearthstone = false;
			Config.Save();
		}
		
		private void CheckboxStartHearthstoneWithHDT_Checked(object sender, RoutedEventArgs e)
		{
			if (!_initialized)
				return;
			Config.Instance.StartHearthstoneWithHDT = true;
			Config.Save();
		}

		private void CheckboxStartHearthstoneWithHDT_Unchecked(object sender, RoutedEventArgs e)
		{
			if (!_initialized)
				return;
			Config.Instance.StartHearthstoneWithHDT = false;
			Config.Save();
		}

		private async void CheckboxConfigSaveAppData_Checked(object sender, RoutedEventArgs e)
		{
#if(!SQUIRREL)
			if(!_initialized)
				return;
			var path = Config.Instance.ConfigPath;
			Config.Instance.SaveConfigInAppData = true;
			XmlManager<Config>.Save(path, Config.Instance);
			await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
			Core.MainWindow.Restart();
#endif
		}

		private async void CheckboxConfigSaveAppData_Unchecked(object sender, RoutedEventArgs e)
		{
#if(!SQUIRREL)
			if(!_initialized)
				return;
			var path = Config.Instance.ConfigPath;
			Config.Instance.SaveConfigInAppData = false;
			XmlManager<Config>.Save(path, Config.Instance);
			await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
			Core.MainWindow.Restart();
#endif
		}

		private async void CheckboxDataSaveAppData_Checked(object sender, RoutedEventArgs e)
		{
#if(!SQUIRREL)
			if(!_initialized)
				return;
			Config.Instance.SaveDataInAppData = true;
			Config.Save();
			await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
			Core.MainWindow.Restart();
#endif
		}

		private async void CheckboxDataSaveAppData_Unchecked(object sender, RoutedEventArgs e)
		{
#if(!SQUIRREL)
			if(!_initialized)
				return;
			Config.Instance.SaveDataInAppData = false;
			Config.Save();
			await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
			Core.MainWindow.Restart();
#endif
		}

		private void CheckboxAdvancedWindowSearch_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.UseAnyUnityWindow = true;
			Config.Save();
		}

		private void CheckboxAdvancedWindowSearch_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.UseAnyUnityWindow = false;
			Config.Save();
		}

		private void ButtonGamePath_OnClick(object sender, RoutedEventArgs e)
		{
			var dialog = new FolderBrowserDialog {Description = "Select your Hearthstone Directory", ShowNewFolderButton = false};
			var dialogResult = dialog.ShowDialog();

			if(dialogResult == DialogResult.OK)
			{
				Config.Instance.HearthstoneDirectory = dialog.SelectedPath;
				Config.Save();
				Core.MainWindow.ShowMessage("Restart required.", "Please restart HDT for this setting to take effect.").Forget();
			}
		}

		private async void SelectSaveDataPath_Click(object sender, RoutedEventArgs e)
		{
#if(!SQUIRREL)
			var dialog = new FolderBrowserDialog();
			var dialogResult = dialog.ShowDialog();

			if(dialogResult == DialogResult.OK)
			{
				var saveInAppData = Config.Instance.SaveDataInAppData.HasValue && Config.Instance.SaveDataInAppData.Value;
				if(!saveInAppData)
				{
					foreach(var value in new List<bool> {true, false})
					{
						Config.Instance.SaveDataInAppData = value;
						Helper.CopyReplayFiles();
						DeckStatsList.SetupDeckStatsFile();
						DeckList.SetupDeckListFile();
						DefaultDeckStats.SetupDefaultDeckStatsFile();
						Config.Instance.DataDirPath = dialog.SelectedPath;
					}
				}
				Config.Instance.DataDirPath = dialog.SelectedPath;
				Config.Save();
				if(!saveInAppData)
				{
					await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
					Core.MainWindow.Restart();
				}
			}
#endif
		}

		private void ButtonOpenAppData_OnClick(object sender, RoutedEventArgs e)
		{
			try
			{
				Process.Start(Config.AppDataPath);
			}
			catch(Exception ex)
			{
				Log.Error(ex);
				ErrorManager.AddError("Could not open AppData folder.", "Manually navigate to '%AppData%/HearthstoneDeckTracker'.");
			}
		}

		private void CheckboxStartWithWindows_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			RegistryHelper.SetRunKey();
			Config.Instance.StartWithWindows = true;
			Config.Save();
		}

		private void CheckboxStartWithWindows_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			RegistryHelper.DeleteRunKey();
			Config.Instance.StartWithWindows = false;
			Config.Save();
		}

		private void CheckboxCheckForBetaUpdates_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CheckForBetaUpdates = true;
			Config.Save();
		}

		private void CheckboxCheckForBetaUpdates_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.CheckForBetaUpdates = false;
			Config.Save();
		}

		private void CheckboxShowSplashScreen_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.ShowSplashScreen = true;
			Config.Save();
		}

		private void CheckboxShowSplashScreen_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.ShowSplashScreen = false;
			Config.Save();
		}

		private void CheckBoxAnalytics_OnChecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.GoogleAnalytics = true;
			Config.Save();
		}

		private void CheckBoxAnalytics_OnUnchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.GoogleAnalytics = false;
			Config.Save();
		}

		private void CheckboxShowNewsBar_OnClick(object sender, RoutedEventArgs e)
		{
			if (!_initialized)
				return;
			ConfigWrapper.IgnoreNewsId = ConfigWrapper.IgnoreNewsId == -1
				? Remote.Config.Data?.News?.Id ?? 0 : -1;
		}

		private void CheckboxAlternativeScreenCapture_Checked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.AlternativeScreenCapture = true;
			Config.Save();
		}

		private void CheckboxAlternativeScreenCapture_Unchecked(object sender, RoutedEventArgs e)
		{
			if(!_initialized)
				return;
			Config.Instance.AlternativeScreenCapture = false;
			Config.Save();
		}

		private async void ButtonHearthstoneLogsDirectory_Click(object sender, RoutedEventArgs e)
		{
			var dialog = new FolderBrowserDialog();
			dialog.SelectedPath = Config.Instance.HearthstoneDirectory;
			var dialogResult = dialog.ShowDialog();

			if (dialogResult == DialogResult.OK)
			{
				//Logs directory needs to be a child directory in Hearthstone directory
				if (!dialog.SelectedPath.StartsWith(Config.Instance.HearthstoneDirectory + @"\"))
				{
					await Core.MainWindow.ShowMessage("Invalid argument", "Selected directory not in Hearthstone directory!");
					return;
				}

				//Check if same path selected (no restart required)
				if (Config.Instance.HearthstoneLogsDirectoryName.Equals(dialog.SelectedPath))
					return;

				Config.Instance.HearthstoneLogsDirectoryName = dialog.SelectedPath.Remove(0, Config.Instance.HearthstoneDirectory.Length + 1);
				Config.Save();

				await Core.MainWindow.ShowMessage("Restart required.", "Click ok to restart HDT");
				Core.MainWindow.Restart();
			}
		}

		private void ButtonDebugWindow_Click(object sender, RoutedEventArgs e)
		{
			new DebugWindow(Core.Game).Show();
		}
	}
}
